import React, { useState, useEffect } from 'react';
import { BACKGROUND_IMAGE_URL, COUNTRIES, WHATSAPP_CHANNEL_URL, GOOGLE_SHEETS_SCRIPT_URL } from './constants';
import { Input } from './components/Input';
import { PhoneInput } from './components/PhoneInput';
import { SocialFooter } from './components/SocialFooter';
import { FormData, FormErrors } from './types';

function App() {
  // State
  const [formData, setFormData] = useState<FormData>({
    firstName: '',
    lastName: '',
    email: '',
    countryIso: 'VE', // Default to Venezuela ISO code
    phoneNumber: '',
  });
  
  const [errors, setErrors] = useState<FormErrors>({});
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isSuccess, setIsSuccess] = useState(false);

  // Auto-detect country on mount
  useEffect(() => {
    const detectCountry = async () => {
      try {
        const response = await fetch('https://ipapi.co/json/');
        if (response.ok) {
          const data = await response.json();
          // We use country_code (ISO) for more precise matching
          const countryIso = data.country_code; // e.g., "US", "VE"
          
          // Verify if it exists in our list to avoid invalid selection
          const supported = COUNTRIES.find(c => c.code === countryIso);
          
          if (supported) {
            setFormData(prev => ({ ...prev, countryIso: supported.code }));
          }
        }
      } catch (error) {
        // Silent fail, stick to default (VE)
        console.warn("Could not detect country automatically");
      }
    };

    detectCountry();
  }, []);

  // Handlers
  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
    // Clear error when user types
    if (errors[name as keyof FormErrors]) {
      setErrors(prev => ({ ...prev, [name]: undefined }));
    }
  };

  const handleCountryChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setFormData(prev => ({ ...prev, countryIso: e.target.value }));
  };

  const validate = (): boolean => {
    const newErrors: FormErrors = {};
    if (!formData.firstName.trim()) newErrors.firstName = 'El nombre es obligatorio';
    if (!formData.lastName.trim()) newErrors.lastName = 'El apellido es obligatorio';
    if (!formData.email.trim()) {
      newErrors.email = 'El correo es obligatorio';
    } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
      newErrors.email = 'Correo inválido';
    }
    
    // Validar numero entero (comprobando longitud minima y que solo sean numeros)
    const cleanPhone = formData.phoneNumber.replace(/\D/g, '');
    if (!cleanPhone) {
      newErrors.phoneNumber = 'El número es obligatorio';
    } else if (cleanPhone.length < 7) {
      newErrors.phoneNumber = 'Ingresa el número completo';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!validate()) return;

    setIsSubmitting(true);

    try {
      // Find country details for submission
      const countryData = COUNTRIES.find(c => c.code === formData.countryIso);
      const countryName = countryData ? countryData.name : 'Unknown';
      const countryPrefix = countryData ? countryData.dial_code : '';

      // If a script URL is provided, try to send data
      if (GOOGLE_SHEETS_SCRIPT_URL) {
        const data = new FormData();
        data.append('Fecha', new Date().toISOString());
        data.append('Nombre', formData.firstName);
        data.append('Apellido', formData.lastName);
        data.append('Email', formData.email);
        
        // Separeted fields for Sheets
        data.append('Pais', countryName);
        data.append('Prefijo', countryPrefix);
        data.append('Telefono', formData.phoneNumber);

        // We use mode 'no-cors' because Google Apps Script web apps do not send CORS headers by default
        await fetch(GOOGLE_SHEETS_SCRIPT_URL, {
          method: 'POST',
          body: data,
          mode: 'no-cors' 
        });
      } else {
        // Simulate delay if no script is connected
        await new Promise(resolve => setTimeout(resolve, 1000));
      }

      setIsSubmitting(false);
      setIsSuccess(true);
      
      // Redirect to WhatsApp channel after short delay
      setTimeout(() => {
        window.open(WHATSAPP_CHANNEL_URL, '_blank');
      }, 1500);

    } catch (error) {
      console.error("Submission error", error);
      setIsSubmitting(false);
      setIsSuccess(true);
      setTimeout(() => {
        window.open(WHATSAPP_CHANNEL_URL, '_blank');
      }, 1500);
    }
  };

  return (
    <div className="min-h-screen w-full relative overflow-x-hidden font-sans text-white bg-black selection:bg-white selection:text-black">
      
      {/* Background Container */}
      <div className="absolute inset-0 z-0 overflow-hidden">
        {/* Animated Background Image - Ken Burns Effect */}
        <div 
          className="absolute inset-0 bg-cover bg-center bg-no-repeat grayscale brightness-[0.5] animate-ken-burns"
          style={{ backgroundImage: `url(${BACKGROUND_IMAGE_URL})`, transformOrigin: 'center center' }}
        />
        
        {/* Animated Particles / Dust Motes */}
        <div className="absolute inset-0 pointer-events-none">
          <div className="absolute top-1/4 left-1/4 w-1 h-1 bg-white rounded-full opacity-20 animate-float" />
          <div className="absolute top-3/4 left-1/3 w-2 h-2 bg-white rounded-full opacity-10 animate-float-delayed" />
          <div className="absolute top-1/2 left-2/3 w-1 h-1 bg-white rounded-full opacity-30 animate-float" style={{ animationDelay: '1s' }} />
          <div className="absolute top-1/4 left-3/4 w-1.5 h-1.5 bg-white rounded-full opacity-15 animate-float-delayed" style={{ animationDelay: '2s' }} />
          <div className="absolute bottom-1/4 right-1/4 w-2 h-2 bg-white rounded-full opacity-10 animate-float" style={{ animationDelay: '3s' }} />
        </div>
      </div>
      
      {/* Dark Overlay for readability */}
      <div className="absolute inset-0 z-0 bg-gradient-to-b from-black/80 via-black/40 to-black/95" />
      
      {/* Main Content */}
      <main className="relative z-10 w-full min-h-screen flex flex-col px-4 md:px-6 pt-8">
        
        {/* Content Wrapper - Centers vertically in the available space */}
        <div className="flex-1 flex flex-col items-center justify-center w-full">
          
          {/* Header Section */}
          <header className="text-center mb-8 md:mb-10 max-w-5xl mx-auto animate-fade-in w-full flex flex-col items-center">
            {/* Title - Gradient, Spaced */}
            <div className="flex flex-col items-center justify-center mb-6 text-center w-full drop-shadow-2xl">
              {/* PRIMACY - Metallic White Gradient */}
              <span className="font-display text-[15vw] sm:text-8xl md:text-9xl lg:text-[11rem] uppercase tracking-[0.05em] bg-gradient-to-b from-white via-zinc-200 to-zinc-500 text-transparent bg-clip-text z-10 leading-[0.85]">
                PRIMACY
              </span>
              {/* BOXING - Darker Metallic Gradient */}
              <span className="font-display text-[15vw] sm:text-8xl md:text-9xl lg:text-[11rem] uppercase tracking-[0.05em] bg-gradient-to-b from-zinc-400 via-zinc-600 to-zinc-800 text-transparent bg-clip-text leading-[0.85] mt-2 md:mt-0">
                BOXING
              </span>
            </div>
            
            {/* Subtitle */}
            <p className="font-sans font-light text-zinc-300 text-sm md:text-lg max-w-lg mx-auto leading-relaxed mt-4 text-center">
              El nuevo evento de boxeo en <span className="font-bold text-white">San Carlos, Cojedes</span>. 
              <span className="block mt-1 text-zinc-400">
                Regístrate para recibir primero toda la información sobre lo que viene.
              </span>
            </p>
          </header>

          {/* Dynamic Content Area */}
          <div className="w-full max-w-[420px] mx-auto relative">
            
            {!isSuccess ? (
              <form 
                onSubmit={handleSubmit}
                // Form with 40% opacity on background
                className="w-full bg-zinc-950/40 backdrop-blur-md border border-white/10 p-6 md:p-8 shadow-2xl animate-slide-up rounded-3xl relative overflow-hidden"
              >
                {/* Subtle light sheen on form */}
                <div className="absolute inset-0 bg-gradient-to-tr from-white/5 to-transparent pointer-events-none" />

                <div className="space-y-4 relative z-10">
                  {/* Stacked on mobile/tablet */}
                  <div className="flex flex-col md:flex-row gap-4">
                    <Input 
                      label="Nombre"
                      name="firstName"
                      value={formData.firstName}
                      onChange={handleChange}
                      error={errors.firstName}
                      placeholder="Tu nombre"
                    />
                    <Input 
                      label="Apellido"
                      name="lastName"
                      value={formData.lastName}
                      onChange={handleChange}
                      error={errors.lastName}
                      placeholder="Tu apellido"
                    />
                  </div>
                  
                  <Input 
                    label="Correo Electrónico"
                    name="email"
                    type="email"
                    value={formData.email}
                    onChange={handleChange}
                    error={errors.email}
                    placeholder="ejemplo@correo.com"
                  />

                  <PhoneInput 
                    countries={COUNTRIES}
                    selectedCountryIso={formData.countryIso}
                    phoneNumber={formData.phoneNumber}
                    onCountryChange={handleCountryChange}
                    onPhoneChange={handleChange}
                    error={errors.phoneNumber}
                  />
                </div>

                <button
                  type="submit"
                  disabled={isSubmitting}
                  className="group w-full mt-8 bg-white text-black font-heading font-bold uppercase text-lg py-4 px-6 tracking-widest hover:bg-zinc-200 transition-all disabled:opacity-50 disabled:cursor-not-allowed rounded-2xl shadow-lg shadow-white/10 relative z-10"
                >
                  <span className="flex items-center justify-center gap-2">
                    {isSubmitting ? (
                      'Procesando...'
                    ) : (
                      'REGISTRARSE'
                    )}
                  </span>
                </button>
                
                <div className="text-center mt-6 relative z-10">
                  <p className="text-zinc-500 text-[10px] uppercase tracking-wider">
                    Serás redirigido al canal de WhatsApp
                  </p>
                </div>
              </form>
            ) : (
              <div className="w-full bg-zinc-950/40 backdrop-blur-md border border-white/10 p-8 text-center shadow-2xl animate-fade-in rounded-3xl flex flex-col items-center justify-center min-h-[300px] relative overflow-hidden">
                 {/* Subtle light sheen */}
                 <div className="absolute inset-0 bg-gradient-to-tr from-white/5 to-transparent pointer-events-none" />

                <div className="w-16 h-16 border border-white/30 rounded-full flex items-center justify-center mb-6 text-white shadow-inner relative z-10">
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-8 w-8" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                  </svg>
                </div>
                <h2 className="font-heading text-3xl font-bold uppercase mb-4 text-white relative z-10">¡Listo!</h2>
                <p className="text-zinc-400 mb-8 font-light text-sm relative z-10">
                  Te estamos redirigiendo a nuestro canal oficial.
                </p>
                
                <a 
                  href={WHATSAPP_CHANNEL_URL}
                  target="_blank" 
                  rel="noopener noreferrer"
                  className="relative z-10 text-xs font-bold uppercase tracking-widest border-b border-zinc-700 pb-1 hover:text-white hover:border-white transition-all text-zinc-500"
                >
                  Ir al canal manualmente
                </a>
              </div>
            )}
          </div>

          {/* Social Links inside content wrapper to keep them with the form */}
          <SocialFooter />
        </div>
        
        {/* Footer Copyright & Contact - Pushed to bottom of screen/container */}
        <footer className="w-full flex flex-col-reverse md:flex-row justify-between items-center gap-4 py-6 mt-8 md:mt-0 border-t border-zinc-900/50 relative z-10">
          <p className="text-zinc-600 text-[10px] uppercase tracking-widest text-center md:text-left">
            &copy; {new Date().getFullYear()} Primacy Boxing. Todos los derechos reservados.
          </p>
          <a href="mailto:contacto@comnitus.com" className="text-zinc-600 hover:text-white transition-colors text-[10px] font-bold tracking-widest uppercase text-center md:text-right">
            contacto@comnitus.com
          </a>
        </footer>
      </main>
    </div>
  );
}

export default App;