import React, { useMemo } from 'react';
import { Country } from '../types';

interface PhoneInputProps {
  countries: Country[];
  selectedCountryIso: string;
  phoneNumber: string;
  onCountryChange: (e: React.ChangeEvent<HTMLSelectElement>) => void;
  onPhoneChange: (e: React.ChangeEvent<HTMLInputElement>) => void;
  error?: string;
}

export const PhoneInput: React.FC<PhoneInputProps> = ({
  countries,
  selectedCountryIso,
  phoneNumber,
  onCountryChange,
  onPhoneChange,
  error,
}) => {
  // Use browser's locale to translate country names
  const localizedCountries = useMemo(() => {
    try {
      const regionNames = new Intl.DisplayNames([navigator.language], { type: 'region' });
      return countries.map(country => ({
        ...country,
        displayName: regionNames.of(country.code) || country.name
      })).sort((a, b) => a.displayName.localeCompare(b.displayName));
    } catch (e) {
      return countries.map(c => ({ ...c, displayName: c.name }));
    }
  }, [countries]);

  // Find currently selected country
  const currentCountry = countries.find(c => c.code === selectedCountryIso);
  const dialCode = currentCountry?.dial_code || '+58';
  const maxDigits = currentCountry?.maxDigits || 15;

  return (
    <div className="w-full">
      <label className="block text-[10px] font-bold tracking-widest text-zinc-500 mb-2 uppercase">
        Número de teléfono / WhatsApp
      </label>
      <div className="flex gap-2 relative z-0">
        <div className="relative w-[35%] md:w-[140px]">
          <select
            name="countryIso"
            value={selectedCountryIso}
            onChange={onCountryChange}
            className="w-full h-full appearance-none bg-zinc-900/50 border border-zinc-800 rounded-2xl text-white pl-3 pr-6 py-3 text-sm focus:outline-none focus:border-white focus:bg-black transition-colors cursor-pointer text-ellipsis overflow-hidden whitespace-nowrap"
          >
            {localizedCountries.map((country) => (
              <option key={country.code} value={country.code} className="text-black bg-white">
                 {country.displayName}
              </option>
            ))}
          </select>
          {/* Custom Arrow - Minimalist */}
          <div className="pointer-events-none absolute inset-y-0 right-0 flex items-center px-2 text-zinc-500">
            <svg className="fill-current h-3 w-3" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20">
              <path d="M9.293 12.95l.707.707L15.657 8l-1.414-1.414L10 10.828 5.757 6.586 4.343 8z" />
            </svg>
          </div>
        </div>
        
        <div className="flex-1 relative">
           {/* Prefix Overlay */}
           <div className="absolute inset-y-0 left-0 pl-4 flex items-center pointer-events-none z-10">
              <span className="text-zinc-400 font-medium text-sm">{dialCode}</span>
           </div>
           
           <input
            type="tel"
            name="phoneNumber"
            placeholder="0000000"
            value={phoneNumber}
            onChange={(e) => {
              const val = e.target.value.replace(/\D/g, '');
              const syntheticEvent = {
                ...e,
                target: {
                  ...e.target,
                  value: val,
                  name: 'phoneNumber'
                }
              };
              onPhoneChange(syntheticEvent as unknown as React.ChangeEvent<HTMLInputElement>);
            }}
            maxLength={maxDigits}
            className={`
              w-full bg-zinc-900/50 border text-white pl-14 pr-4 py-3 rounded-2xl text-sm
              placeholder-zinc-700 focus:outline-none focus:border-white focus:bg-black
              transition-all duration-300
              ${error ? 'border-red-900 focus:border-red-500' : 'border-zinc-800'}
            `}
          />
        </div>
      </div>
      {error && (
        <p className="text-red-500 text-[10px] mt-1.5 font-medium tracking-wide">{error}</p>
      )}
    </div>
  );
};